## Function to compute the ESS of the prior distribution for w=log[pC/(1-pC)] when prior distributions incorporate related historical data
## Function inputs:	a, b = parameters of Beta(a,b) prior distribution for pC reflecting what is known without the MYCYC data
##						lCparam, lEparam = parameters (mean, variance) of normal prior distributions for lambdaC and lambdaE
##						sigmainv, musigma = reciprical of SD and mean/SD of normal prior distribution for theta reflecting what is known without the MYCYC data
##						mycyc = vector containing summaries of mycyc data
##						meanpc = expectation of prior distribution for pC (incorporating MYCYC data)
##						normEC = normalising constant of prior density for (pE, pC)|zr, where zr = mycyc data					
## Function returns: effective sample size of prior distribution log[pC/(1-pC)]|zr

D2ess_pc <- function(a, b, lCparam, lEparam, sigmainv, musigma, mycyc, meanpc, normEC){

  	## Calculating prior variance of omega = log[pC/(1-pc)] 
  	## Create a grid for omega assuming it is approximately normally distributed. 	
  	varo = D2logoddspc(a,b, lCparam, lEparam, sigmainv, musigma, mycyc, normEC, meanpc)
  	r = as.integer(32)
   	mesh = as.integer(6*r -1)
   	mesh1 = as.integer(2*mesh-1)
   	grid1 = vector(mode="numeric", length= mesh)
   	grido = vector(mode="numeric", length= mesh1)
   	## centre mesh at log(E(pc)/(1-E(pc))), where E(pc) is prior expectation of pC incorporating MYCYC data
  	mu = log(meanpc/(1-meanpc))
   	for(i in 1:mesh){
   		if(i <= (r-1)){
   			grid1[i] = mu + sqrt(varo)*(-3-4*log(r/i))
   		}else if((i >= r) & (i<= 5*r)){
   			grid1[i] = mu + sqrt(varo)*(-3 + 3*(i-r)/(2*r))
   		}else{
   			grid1[i] = mu + sqrt(varo)*(3+ 4*log(r/(6*r -i)))
   		}
   	}   
	## calculating mesh mid-points 
   	for(i in seq(1, mesh, by=1)){
   		grido[2*i-1] = grid1[i]	
   	}
   	for(i in seq(2, mesh1-1, by=2)){
   		grido[i] = (grido[i+1] + grido[i-1])/2.0
   	}    
	## calculating Simpson's integration weights
   	womega  = vector(mode="numeric", length=mesh1)
    womega[1] = (grido[3]-grido[1])/6.0
  	womega[mesh1] = (grido[mesh1] - grido[mesh1-2])/6.0
  	for(i in seq(2, (mesh1-1), by=2)){
    	womega[i] = 4*(grido[i+1] - grido[i-1])/6.0
   	}
   	for(i in seq(3, (mesh1-2), by=2)){
		womega[i] = (grido[i+2] - grido[i-2])/6.0
   	}
   	
   	## Evaluating integral of f_{omega}(omega)*exp(omega)/((1 + exp(omega))^2
   	dens1 = D2pc_odds(grido, a, b, lCparam, lEparam, sigmainv, musigma, mycyc, normEC)
   	const1 = exp(grido)/((1 + exp(grido))^2)
   	expect = sum(womega*dens1*const1)
   	neff = 1/(varo*expect)  

  return(neff)
}

## Function to compute the ESS of the prior distribution of theta incorporating the related historical dataset.
## Function inputs:	R = number of subjects randomised to MMF for every one on CYC in the hypothetical trial which we used to calculate ESS (in the paper we always set R=1)
##					mu, sigma2 = parameters of prior distribution of theta N(mu, sigma2) reflecting what is known without the MYCYC data
##					a, b = parameters of Beta(a,b) prior distribution for pC reflecting what is known without the MYCYC data
##					lCparam, lEparam = parameters (mean, variance) of normal prior distributions for lambdaC and lambdaE
##					mycyc = vector containing summaries of mycyc data
##					normEC = normalising constant of prior density for (pE, pC)|zr, where zr = mycyc data	
##					vart = variance of prior distribution of theta incorporating mycyc data
## Function returns:ESS of prior distribution for theta (in terms of total number of effective observations accumulated across E and C)

D2ess_theta <- function(R, mu, sigma2, a, b, lCparam, lEparam, mycyc, normEC, vart){	

    gridc = seq(0.001, 0.999, by=0.001)
    midp1 = (0.00001 + 0.001)*0.5
    midp2 = (0.99999 + 0.999)*0.5
    gridc = append(c(0.00001, midp1), gridc)
    gridc = append(gridc, c(midp2, 0.99999))
  	lc = length(gridc)

  	wc  = vector(mode="numeric", length=lc)
  	wc[1] = (gridc[3]-gridc[1])/6.0
  	wc[lc] = (gridc[lc] - gridc[lc-2])/6.0
  	for(i in seq(2, lc-1, by=2)){
    	wc[i] = 4*(gridc[i+1] - gridc[i-1])/6.0
  	}
  	for(i in seq(3, lc-2, by=2)){
    	wc[i] = (gridc[i+2] - gridc[i-2])/6.0
  	}
  	## Setting up similar grid for pE
  	gride = gridc
  	gridea = gridc
  	gridca = gridc
  	we = wc
  	le = lc

  	expect = vector(mode="numeric", le)
	denspc = vector(mode="numeric", length=lc)
    intpca =vector(mode = "numeric", length=lc)
    
  	## Integrating over the joint density for (pE, pC|zr) 	 	
  	for(j in 1:lc){
    	const1 = dnorm(log(gridca*(1-gridc[j])/(gridc[j]*(1-gridca))), mean = lCparam[1], sd = sqrt(lCparam[2]))
    	dens = (gridca^(mycyc[1]-1))*((1 - gridca)^(mycyc[2]-1))*const1
    	intpca[j] = sum(wc*dens)
    }
    for(i in 1:le){
    	const1 = dnorm(log(gridea*(1-gride[i])/(gride[i]*(1-gridea))), mean = lEparam[1], sd = sqrt(lEparam[2]))
    	dens = (gridea^(mycyc[3] - 1))*((1-gridea)^(mycyc[4]-1))*const1
		intpea = sum(wc*dens)
    	for(j in 1:lc){
    		## Integrating across the joint density of (pE, pC)|zr
    		const1 = dnorm(log(gride[i]*(1-gridc[j])/(gridc[j]*(1 - gride[i]))), mean = mu, sd = sqrt(sigma2))
    		pbar = (R*gride[i] + gridc[j])/(R+1)
			denspc[j] = pbar*(1-pbar)*normEC*const1*intpca[j]*intpea*dbeta(gridc[j], shape1=a, shape2=b)/(gride[i]*(1-gride[i]))
		}
		expect[i] = sum(wc*denspc)
	}
  	const1 = sum(we*expect)*R/((R+1)^2)
	neff = 1/(vart*const1)
	return(neff)
}

## Function to calculate variance of prior distribution of log[pc/(1-pc)] given related dataset
## Function inputs: 	a, b = parameters of Beta(a,b) prior distribution for pC reflecting what is known without the MYCYC data
##						lCparam, lEparam = parameters (mean, variance) of normal prior distributions for lambdaC and lambdaE
##						sigmainv, musigma = reciprical of SD and mean/SD of normal prior distribution for theta reflecting what is known without the MYCYC data
##						mycyc = vector containing summaries of mycyc data
##						normEC = normalising constant of prior density for (pE, pC)|zr, where zr = mycyc data
##						meanpc = expectation of prior distribution for pC (incorporating MYCYC data)		
## Function returns: variance of prior distribution of log odds of success on CYC given the related dataset zr

D2logoddspc <- function(a,b, lCparam, lEparam, sigmainv, musigma, mycyc, normEC, meanpc){
	
    ## set up a grid for omega = log[pc/(1-pc)] assuming approximately normally distributed
    r = as.integer(64)
   	mesh = as.integer(6*r -1)
   	mesh1 = as.integer(2*mesh-1)
   	grid1 = vector(mode="numeric", length= mesh)
   	grido = vector(mode="numeric", length= mesh1)
   	
   	## centre mesh at log[E(pc)/(1-E(pc))], where E(pc) is prior mean of pC|related data 
    mu = log(meanpc/(1-meanpc)) 	
   	for(i in 1:mesh){
   		if(i <= (r-1)){
   			grid1[i] = mu + sqrt(3.0)*(-3-4*log(r/i))
   		}else if((i >= r) & (i<= 5*r)){
   			grid1[i] = mu + sqrt(3.0)*(-3 + 3*(i-r)/(2*r))
   		}else{
   			grid1[i] = mu + sqrt(3.0)*(3+ 4*log(r/(6*r -i)))
   		}
   	}   
 
   	for(i in seq(1, mesh, by=1)){
   		grido[2*i-1] = grid1[i]	
   	}
   	for(i in seq(2, mesh1-1, by=2)){
   		grido[i] = (grido[i+1] + grido[i-1])/2.0
   	}    
	## calculating Simpson's integration weights
   	womega  = vector(mode="numeric", length=mesh1)
    womega[1] = (grido[3]-grido[1])/6.0
  	womega[mesh1] = (grido[mesh1] - grido[mesh1-2])/6.0
  	for(i in seq(2, (mesh1-1), by=2)){
    	womega[i] = 4*(grido[i+1] - grido[i-1])/6.0
   	}
   	for(i in seq(3, (mesh1-2), by=2)){
		womega[i] = (grido[i+2] - grido[i-2])/6.0
   	}

   	dens1 = D2pc_odds(grido, a, b, lCparam, lEparam, sigmainv, musigma, mycyc, normEC)
    expect = sum(grido*womega*dens1)
    var1 = sum(womega*grido*grido*dens1) - (expect^2)
    
    return(var1)
}